const fs = require('fs');

// setzen der globalen datenbank wenn nicht definiert
if (typeof __database === 'undefined')
    global.__database = false;

// lädt die datenbank
const loadDB = async () => {
    return new Promise(async (resolve, reject) => { // wir geben ein "versprechen" zurück
        //nur laden wenn noch nicht geladen wurde
        if (__database === false) {
            let databaseData = {};
            try { // wir versuchen die datei zu lesen
                let data = fs.readFileSync('./database.json', { encoding: "utf8" }); // datei inhalt lesen
                databaseData = JSON.parse(data); // parse data
            } catch (err) { // wenn ein fehler auftritt
                if (err.code !== 'ENOENT') { // wenn es NICHT der "datei existiert nicht" fehler ist
                    console.error(err);
                    reject(err); // lehne "versprechen" ab
                }
            }

            __database = databaseData;
        }

        resolve('success'); // erfüllt "versprechen"
    });
}

// speichert die "datenbank"
const saveDB = async () => {
    const content = JSON.stringify(__database); // umwandeln der datenbank in einen json string
    return new Promise(async (resolve, reject) => {
        try { // wir versuchen zu speichern
            fs.writeFileSync('./database.json', content, { encoding: "utf8" });
        } catch (err) { // speichern in datei
            console.error(err);
            reject(err);
        };

        resolve('success');
    });
}

module.exports = {
    // gibt eine value des "tabelleneintrags" zurück
    // table: string = name der "tabelle"
    // key: string = schlüssel des eintrags
    // def: any = default wert wenn nicht vorhanden
    get: async (table, key, def) => {
        return new Promise(async (resolve) => { // wir geben ein "versprechen" zurück
            await loadDB(); // lädt die datenbank wenn noch nicht geladen

            if (typeof __database[table] !== 'undefined') // prüfen ob die tabelle exisitert
                resolve(((typeof __database[table][key] !== 'undefined') ? __database[table][key] : def)); // rückgabe des wertes wenn vorhanden sonst default

            resolve(def); // rückgabe des default wertes ( wir erfüllen immer das versprechen )
        });
    },

    // speichert einen wert in die "tabelle" der "datenbank"
    // table: string = name der "tabelle"
    // key: string = schlüssel des eintrags
    // value: any = wert des eintrags 
    set: async (table, key, value) => {
        return new Promise(async (resolve) => { // wir geben ein "versprechen" zurück
            await loadDB(); // lädt die datenbank wenn noch nicht geladen

            if (typeof __database[table] === 'undefined') // definieren der "tabelle" wenn nicht existiert
                __database[table] = {};

            __database[table][key] = value; // setzen des values

            await saveDB(); // speichern der DB
            
            resolve(value); // rückgabe des neuen wertes ( wir erfüllen immer das versprechen )
        });
    },

    // addiert X zu einem wert und speichert es in die db
    // table: string = name der "tabelle"
    // key: string = schlüssel des eintrags
    // inc: int = um wie viel wird erhöt  
     // max: int|false = bestimmt das maximum ( default keins ) 
    increase: async (table, key, inc, max = false) => {
        return new Promise(async (resolve) => { // wir geben ein "versprechen" zurück
            let value = await module.exports.get(table, key, 0);
            value     = await module.exports.set(table, key, value + inc);

            if (max !== false && value > max)
                value = max;

            resolve(value); // wir erfüllen dieses versprechen
        });
    },

    // subtrahiert X von einem wert und speichert es in die db
    // table: string = name der "tabelle"
    // key: string = schlüssel des eintrags
    // dec: int = um wie viel wird vermindert 
    // min: int|false = bestimmt das minimum ( default keins ) 
    decrease: async (table, key, dec, min = false) => {
        return new Promise(async (resolve) => { // wir geben ein "versprechen" zurück
            let value = await module.exports.get(table, key, 0);
            value     = await module.exports.set(table, key, value - dec);

            if (min !== false && value < min)
                value = min;
            resolve(value); // wir erfüllen dieses versprechen
        });
    }
}